//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.js
//
/*global devTime, alert, handleError*/
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path="~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>

(function ($, window) {

    var regexPlus = /\+/g,
        utcOffset = 0,
        uiCulture,
        debug = false;

    if (typeof _globalDebugFlag !== "undefined") {
        debug = _globalDebugFlag;
    }

    if (typeof __utcOffset !== "undefined") {
        utcOffset = __utcOffset || 0;
    }

    if (typeof __uiCulture !== "undefined") {
        uiCulture = __uiCulture;
    }

    if (typeof TFS === 'undefined') {
        TFS = (function () {

            var modules = {},
                plugins = {},
                scripts = {},
                moduleLoadStates = {},
                unloadRequested = false,
                config = {
                    base: "/_scripts/",
                    flavor: ".debug."
                },
                errorHandler,
                bases,
                moduleExceptions = {
                    ServerException: "TFS.ServerException"
                };


            Error.serverError = function (error) {
                var err = Error.create((error && error.message) || TFSSeedFileResources.UnknownServerErrorMessage, $.extend(error || {}, { name: moduleExceptions.ServerException }));
                err.popStackFrame();

                return err;
            };

            Error.circularDependency = function (moduleName, moduleRef) {
                var err = Error.create(String.format(TFSSeedFileResources.CircularDependencyErrorFormat, moduleName, moduleRef), { moduleName: moduleName, moduleReference: moduleRef });
                err.popStackFrame();

                return err;
            };

            function configure() {
                var option = arguments[0];
                if (typeof option === "undefined") {
                    return config;
                }
                else if (typeof option === "string") {
                    config[option] = arguments[1];
                }
                else if (typeof option === "object") {
                    config = $.extend(config, option);
                }
            }

            function getModuleBase(moduleName) {
                var base, pos;

                if (bases) {
                    base = bases[moduleName];

                    if (!base) {
                        pos = moduleName.lastIndexOf(".");
                        if (pos > 0) {
                            return getModuleBase(moduleName.substring(0, pos));
                        }
                    }
                }

                return base || config.base;
            }

            function setNamespaceBase(namespace, base) {
                if (base) {
                    bases = bases || {};
                    bases[namespace] = base;
                }
            }

            function getModuleUrl(moduleName) {
                return getModuleBase(moduleName) + moduleName + config.flavor + "js?loc=" + encodeURIComponent(uiCulture || Sys.CultureInfo.CurrentCulture.name);
            }

            function delegateAfterNTimes(context, n, callback, args) {
                var i = 0;
                return function () {
                    if (++i >= n) {
                        return callback.apply(context || TFS, args || []);
                    }
                };
            }

            function queueCallbacks(context, callback, errorCallback, data) {
                var callbacks = {}, initialCookie, cbCount = 0, callbackCookie = 0, resultArgs, finished = false, failed = false;

                function registerCallbacks(callback, errorCallback, data) {
                    var cookie;
                    if (callback || errorCallback) {
                        cookie = callbackCookie++;
                        cbCount++;
                        callbacks[cookie] = [callback, errorCallback, data];
                    }

                    return cookie;
                }

                initialCookie = registerCallbacks(callback, errorCallback, data);

                function notify() {
                    var cbCookie, cbRecord, cbIndex = finished ? 0 : (failed ? 1 : -1), cb, cbData, handled = false;

                    if (cbIndex >= 0) {
                        for (cbCookie in callbacks) {
                            if (callbacks.hasOwnProperty(cbCookie)) {
                                cbRecord = callbacks[cbCookie];

                                cb = cbRecord[cbIndex];
                                if (cb) {
                                    cbData = cbRecord[2];
                                    cb.apply(context, (resultArgs || []).concat([cbData]));
                                    handled = true;
                                }
                            }
                        }

                        callbacks = {};
                        cbCount = 0;
                    }

                    return handled;
                }

                return {
                    cookie: initialCookie,
                    count: function () {
                        return cbCount;
                    },
                    finish: function () {
                        finished = true;
                        resultArgs = Array.prototype.slice.call(arguments, 0);

                        notify();
                    },
                    error: function () {
                        failed = true;
                        resultArgs = Array.prototype.slice.call(arguments, 0);
                        if (resultArgs.length === 0) {
                            resultArgs = [Error.invalidOperation(TFSSeedFileResources.UnknownServerErrorMessage)];
                        }

                        if (!notify()) {
                            handleError(resultArgs[0], null, context);
                        }
                    },
                    register: function (callback, errorCallback, data) {
                        var cookie = registerCallbacks(callback, errorCallback, data);

                        if (cookie) {
                            notify();
                        }

                        return cookie;
                    },
                    unregister: function (cookie) {
                        cbCount--;
                        delete callbacks[cookie];
                    }
                };
            }


            function queueRequest(context, target, propName, successCallback, errorCallback, worker) {
                /// <summary>
                /// Queues a request for a piece of data.  Handles situations where the data has already been
                /// retrieved as well as when multiple requests are pending for the same data.  When the data has
                /// already been retrieved, the successCallback will be invoked immediately.  When multiple
                /// requests are pending for the same data, each of the callers will be notified when the data
                /// request has been completed (worker will only be invoked once).
                ///
                /// Sample usage:  This will invoke the worker function using the current object as the context.  The "_teamSettings"
                ///                property of the current object will be checked for a value before invoking the worker.  If the value
                ///                needs to be looked up, the worker will be invoked and it will make a request for the data.  If the
                ///                request is completed successfully the data is passed to the succeeded callback.  If there is an error
                ///                with the request the failed callback is invoked.
                ///
                ///     queueRequest(this, this, "_teamSettings", successCallback, errorCallback,
                ///         function (succeeded, failed) {
                ///             var url = that.getTfsContext().getActionUrl("teamSettings", "wit", { area: "api" });
                ///
                ///             Ajax.getMSJSON(url, null, function (teamSettings) {
                ///                 succeeded(teamSettings);
                ///             }, failed);
                ///         });
                ///
                /// </summary>
                /// <param name="context" type="object">The "this" that the worker and successCallback functions will be invoked with.</param>
                /// <param name="target" type="object">
                /// The object which the propName property should be checked on to see if the request has already been performed.
                /// If the property has a value (that is not a function), then the success callback is invoked immediately with the properties value as the result.
                /// If the property does not have a value, the request is processed and the result is stored in the property.
                /// </param>
                /// <param name="propName" type="String">Name of the property on the target to store the result in and check to see if the request is needed.</param>
                /// <param name="successCallback" type="function">Function invoked when the request is completed.  The function should take the "result" as its first parameter.</param>
                /// <param name="errroCallback" type="function">Function invoked when the request has completed with an error. The function should take the "error" as its first parameter.</param>
                /// <param name="worker" type="function">
                /// This is the which performs the work to retrieve the data.  The function should have the following signature:
                ///     function worker(succeeded, failed)
                ///
                /// The worker should invoke the "succeeded" function that it is provided passing it the result.  If an error occurs the worker should invoke the
                /// "failed" function with the error.
                ///
                /// NOTE: It is important to note that the "successCallback" is not the same as the "succeeded" callback provided to the worker
                ///       function.  It is important for the worker to invoke the callbacks it is provided with rather than the callbacks which are
                ///       provided to the queueRequest method.  The same holds true for the failed callback provided to the worker function.
                /// </param>

                var result = target[propName], queue;

                function succeeded(result) {
                    target[propName] = result;
                    queue.finish(result);
                }

                function failed(error) {
                    target[propName] = null;
                    queue.error(error);
                }

                if (worker && result === null || typeof result === "undefined") {
                    queue = queueCallbacks(context, successCallback, errorCallback);
                    target[propName] = queue.register;

                    worker.call(context, succeeded, failed);
                }
                else if ($.isFunction(result)) {
                        //result is our queue register function. add our callbacks to the queue
                    result(successCallback, errorCallback);
                }
                else if ($.isFunction(successCallback)) {
                    successCallback.call(context, result);
                }


            }

            function getErrorMessage(error) {
                if (error) {
                    if (typeof error === "string") {
                        return error;
                    }
                    else if ($.isFunction(error)) {
                        return getErrorMessage(error());
                    }
                    else if (error.message) {
                        return error.message;
                    }
                    else {
                        return error.toString();
                    }
                }
                else {
                    return TFSSeedFileResources.UnknownErrorMessage;
                }
            }

            // TODO: Think about moving this to a more suitable place
            function ErrorHandler() {
                /// <summary>Global error handler class which is attached to TFS</summary>
            }

            ErrorHandler.prototype = {
                $error: null,
                visible: false,

                initialize: function () {
                    /// <summary>(Internal function) Initializes error handler</summary>

                    var that = this;
                    this.$error = $(".error-section").bind("keydown", function (e) {
                        // Closing error panel when escape key is pressed
                        if (e.keyCode === 27) { // escape key
                            that.hideError();
                            return false;
                        }
                    });
                },
                exists: function () {
                    /// <summary>(Internal function) Checks whether error container exists or not</summary>
                    return this.$error.length > 0;
                },
                showError: function (message, status, stackTrace) {
                    /// <summary>(Internal function) Shows error in the container</summary>
                    var that = this, $c, $li;

                    function $createErrorElement($parent) {
                        var msg = "";
                        if (status) {
                            msg = status + ": ";
                        }

                        if (message) {
                            msg = msg + message;
                        }

                        $li = $("<li>").text(msg).appendTo($parent);
                        if (stackTrace) {
                            $("<pre>").text(stackTrace).appendTo($li);
                        }
                    }

                    if (!this.visible) {
                        this.visible = true;

                        this.$error.empty();

                        // Creating close link
                        $("<div tabIndex=0 />").addClass("close").text(TFSSeedFileResources.ErrorPaneCloseLink).appendTo(this.$error)
                            .click(function () {
                                that.hideError();
                            })
                            .keydown(function (e) {
                                // Closing dialog when enter key is pressed when this div has the focus
                                if (e.keyCode === 13) { // enter key
                                    that.hideError();
                                    return false;
                                }
                            });

                        // Creating message panel
                        $c = $("<div>").addClass("message").appendTo(this.$error);
                        $("<h4>").text(TFSSeedFileResources.ErrorPaneHeader).appendTo($c);
                        $("<ul>").addClass("error-list").appendTo($c);
                    }

                    $createErrorElement($("ul.error-list", this.$error));

                    this.$error.show()
                        .focus(); // Setting the focus in case user wants to close this error panel by hitting esc key
                },
                hideError: function () {
                    /// <summary>(Internal function) Hides the error when clicked</summary>
                    $("p", this.$error).remove();
                    this.$error.hide();
                    this.visible = false;
                },
                show: function (error) {
                    /// <summary>Displays error in a container. If no container is found, error
                    /// message is displayed in an alert dialog</summary>
                    if (this.exists()) {
                        this.showError(error.message, error.status, error.stack);
                    }
                    else {
                        alert(error.message);
                    }
                }
            };

            errorHandler = new ErrorHandler();

            function handleError(error, callback, context) {
                if ($.isFunction(callback)) {
                    callback.call(context, error);
                }
                else {
                    errorHandler.show(error);
                }
            }

            // TODO: Think about moving this to a more suitable place
            function GlobalProgressIndicator() {
                /// <summary>Global handler for displaying progress during page loads, module loads, ajax requests, or any other registered long-running operations</summary>

                var that = this;
                this._progressPendingActions = {};
                this._pageProgressElements = [];

                $(function () {
                    // Find indicators on the page (by class) on page ready
                    var pageProgressIndicators = $(".pageProgressIndicator");
                    if (pageProgressIndicators.length > 0) {
                        that._addProgressElement(pageProgressIndicators);
                    }
                });
            }

            GlobalProgressIndicator.prototype = {

                _progressPendingActions: null,
                _progressPendingActionsCount: 0,
                _progressPendingActionsNewId: 0,
                _pageProgressElements: null,
                _pageProgressDelayShowTimeout: null,
                _pageProgressMinShowTimeout: null,
                _showingProgress: false,

                getProgressElements: function () {
                    return this._pageProgressElements;
                },

                registerProgressElement: function (element) {
                    this._addProgressElement(element);
                    if (this._showingProgress) {
                        element.show();
                    }
                    else {
                        element.hide();
                    }
                },

                unRegisterProgressElement: function (element) {
                    if (this._pageProgressElements) {
                        this._pageProgressElements = this._pageProgressElements.subtract([element]);
                    }
                },

                _addProgressElement: function (element) {
                    this._pageProgressElements.push(element);
                },

                _showProgressElements: function () {
                    var that = this, progressElements = this._pageProgressElements;

                    if (this._progressPendingActionsCount === 0) {
                        return;
                    }

                    this._showingProgress = true;

                    if (progressElements.length === 0) {
                        return;
                    }

                    $.each(progressElements, function (i, element) {
                        element.show();
                    });

                    if (!this._pageProgressMinShowTimeout) {
                        this._pageProgressMinShowTimeout = window.setTimeout(function () {
                            that._pageProgressMinShowTimeout = null;
                            if (that._progressPendingActionsCount === 0) {
                                that._hideProgressElements();
                            }
                        }, 250);
                    }
                },

                _hideProgressElements: function () {
                    this._showingProgress = false;
                    $.each(this._pageProgressElements, function (i, element) {
                        element.hide();
                    });
                },

                actionStarted: function (name, immediate) {
                    var id, that = this;

                    id = ++this._progressPendingActionsNewId;
                    this._progressPendingActions[id] = name;

                    if (this._progressPendingActionsCount++ === 0) {
                        if (immediate === true) {
                            if (this._pageProgressDelayShowTimeout) {
                                window.clearTimeout(this._pageProgressDelayShowTimeout);
                                this._pageProgressDelayShowTimeout = null;
                            }
                            this._showProgressElements();
                        }
                        else if (!this._pageProgressDelayShowTimeout) {
                            this._pageProgressDelayShowTimeout = window.setTimeout(function () {
                                that._pageProgressDelayShowTimeout = null;
                                that._showProgressElements();
                            }, 250);
                        }
                    }

                    return id;
                },

                actionCompleted: function (id) {
                    delete this._progressPendingActions[id];
                    if (--this._progressPendingActionsCount === 0 && !this._pageProgressMinShowTimeout) {
                        this._hideProgressElements();
                    }
                },

                getPendingActions: function () {
                    var actionNames = [];
                    $.each(this._progressPendingActions, function (id, name) {
                        actionNames.push(id + ": " + name);
                    });
                    return actionNames;
                }
            };

            globalProgressIndicator = new GlobalProgressIndicator();

            $(window).bind("beforeunload", function () {
                unloadRequested = true;
            });

            (function () {
                var search = /^(.*)TFS(\.[^\.]+\.)js(\?.*)?$/;
                $("script").each(function () {
                    var match, src = $(this).attr("src") || "";
                    match = src.match(search);

                    if (match) {
                        scripts[src.toUpperCase()] = true;
                        configure({ src: src, base: match[1], flavor: match[2] });
                        return false;
                    }
                });
            }());

            $(function () {
                var disabled = [], actionId;

                actionId = TFS.globalProgressIndicator.actionStarted("Initial page load", true);

                if (typeof _disabledPlugins !== "undefined") {
                    disabled = _disabledPlugins;
                }

                if (typeof _bases !== "undefined" && _bases.length) {
                    $.each(_bases, function (i, b) {
                        setNamespaceBase(b.namespace, b.base);
                    });
                }

                if (typeof _plugins !== "undefined" && _plugins.length) {
                    $.each(_plugins, function (i, p) {
                        if (!disabled.contains(p.namespace)) {
                            TFS.pluginModule(p.namespace, p.loadAfter);
                        }
                    });
                }

                if (typeof _builtinPlugins !== "undefined" && _builtinPlugins.length) {
                    $.each(_builtinPlugins, function (i, p) {
                        if (!disabled.contains(p.namespace)) {
                            TFS.pluginModule(p.namespace, p.loadAfter);
                        }
                    });
                }

                errorHandler.initialize();
                TFS.globalProgressIndicator.actionCompleted(actionId);
            });

            return $.extend({
                config: configure,
                namespace: function (namespace, body) {
                    var nameParts = namespace.split("."), i, root = window, ns, last, prop;

                    for (i = 0; i < nameParts.length; i++) {
                        last = i === nameParts.length - 1;
                        ns = root[nameParts[i]];

                        if (!ns) {
                            ns = root[nameParts[i]] = last ? (body || {}) : {};
                        }
                        else if (last && body) {
                                // Extending current namespace. We might use $.extend here but
                                // we don't want to take much dependency on jQuery for this file
                            for (prop in body) {
                                if (body.hasOwnProperty(prop)) {
                                    ns[prop] = body[prop];
                                }
                            }
                        }

                        root = ns;
                    }

                    return ns;
                },
                hasUnloadRequest: function () {
                    return unloadRequested;
                },
                getModuleLoadStates: function () {
                    return moduleLoadStates;
                },
                getModuleBase: getModuleBase,
                getScript: function (moduleName, callback) {
                    var url = getModuleUrl(moduleName), src;
                    src = url.toUpperCase();

                    $(function () {
                        var cbQueue = scripts[src], head, script, done, actionId;

                        if (cbQueue) {
                            if ($.isFunction(callback)) {
                                if ($.isFunction(cbQueue)) {
                                    cbQueue(callback);
                                }
                                else {
                                    callback();
                                }
                            }
                        }
                        else {
                            cbQueue = queueCallbacks(TFS, callback);
                            scripts[src] = cbQueue.register;

                            actionId = TFS.globalProgressIndicator.actionStarted("Load script module: " + moduleName);

                            head = document.getElementsByTagName("head")[0] || document.documentElement;
                            script = document.createElement("script");
                            done = false;

                            script.onload = script.onreadystatechange = function () {
                                var module, timerId;
                                if (!done && (!this.readyState || this.readyState === "loaded" || this.readyState === "complete")) {
                                    done = true;

                                    scripts[src] = true;
                                    if (modules[moduleName]) {
                                        cbQueue.finish();
                                    }

                                    // Handle memory leak in IE
                                    script.onload = script.onreadystatechange = null;
                                    script.onerror = null;

                                    if (head && script.parentNode) {
                                        head.removeChild(script);
                                    }

                                    if (!modules[moduleName] && !unloadRequested) {
                                        Sys.Debug.trace(String.format("Module '{0}' cannot be loaded. It may contain script errors.", moduleName));
                                        errorHandler.show(Error.create(String.format(TFSSeedFileResources.ModuleScriptErrorFormat, moduleName)));
                                    }

                                    TFS.globalProgressIndicator.actionCompleted(actionId);
                                }
                            };

                            script.src = url;

                            head.insertBefore(script, head.firstChild);
                        }
                    });
                },
                _loadModuleNamespace: function (moduleName) {
                    /// <summary>Completes the initilization of the module by attaching the contents of the module to the namespace provided for the module.</summary>
                    /// <param name="moduleName" type="String">Name of the module to complete the attach for.</param>

                    var module = modules[moduleName] || {},
                        moduleResult = {};

                    // If the module has not already been fully attached, then complete the attach process.
                    if (!module.attached) {
                        try {
                            // If the module has a body, get the contents of the module.
                            if (module.body != null) {
                                // The module body can be a function or an object.
                                if (typeof module.body === "function") {
                                    // Execute the function to get the module contents.
                                    moduleResult = module.body.call(TFS);
                                }
                                else {
                                    // Use the body directly as the contents of the module.
                                    moduleResult = module.body;
                                }
                            }

                            moduleLoadStates[moduleName] = "loaded";
                        }
                        catch (e) {
                            if (debug) {
                                Sys.Debug.trace(String.format("Module '{0}' cannot be loaded because of an error. {1}", moduleName, getErrorMessage(e)));
                            }

                            moduleLoadStates[moduleName] = "error";

                            throw e;
                        }

                        if (debug) {
                            Sys.Debug.trace(String.format("Module '{0}' loaded.", moduleName));
                        }

                        // Flag the module as attached so we do not attempt to attach again.
                        module.attached = true;

                        // Setup the contents of the module in the namespace provided for the module.
                        TFS.namespace(module.namespace, moduleResult);

                        modules[moduleName] = module;
                    }
                },

                attachModule: function (moduleName, missingCallback, dependencyHash, visitedHash, loadAfter) {
                    /// <summary>Attempt to perform the attach of the module with the provided name.</summary>
                    /// <param name="missingCallback" type="function">
                    /// OPTIONAL: Function to invoke if the module is missing.  The function will be invoked with the name of the missing module.
                    /// </param>
                    /// <param name="dependencyHash" type="object">
                    /// OPTIONAL: Used internally by the attachModule method to keep track of the modules which are being loaded as part of a dependency chain.  If not
                    /// provided this parameter will be created automatically and passed to subsequent attachModule calls that are generated from this one.  Used in
                    /// detecting circular dependencies.
                    /// </param>
                    /// <param name="loadAfter" type="String">
                    /// OPTIONAL: Used internially by the attachModule method to indicates that this module should not be initialized until after the loading of the
                    /// "loadAfter" module has completed.
                    /// </param>

                    var i, l,
                        usings,
                        module = modules[moduleName],
                        result = true,
                        subResult,
                        referencedModule,
                        attachedPlugins,
                        plugin,
                        pluginDependencyHash;

                    // If the module is already attached, then all of its dependencies are already loaded
                    // and initialized, so there is nothing to do here.
                    if (module && module.attached) {
                        return true;
                    }

                    visitedHash = visitedHash || {};

                    if (moduleName in visitedHash) {
                        return visitedHash[moduleName];
                    }

                    visitedHash[moduleName] = false;

                    attachedPlugins = plugins[moduleName];
                    dependencyHash = dependencyHash || {};

                    if (!moduleLoadStates[moduleName]) {
                        moduleLoadStates[moduleName] = "loading";
                    }

                    // If we have the module, ensure that its dependencies and associated plugins
                    // are loaded and initialize the module.
                    if (module) {
                        try {
                            // Track the names of modules which are being loaded as part of this dependency chain.  This is
                            // used in detecting circular dependencies.
                            dependencyHash[moduleName] = true;
                            usings = module.using;

                            // Load the dependencies.
                            if (usings && usings.length) {
                                for (i = 0, l = usings.length; i < l; i++) {
                                    referencedModule = usings[i];

                                    // If the dependend module is already in the depencency hash it indicates that a
                                    // circular dependency exists.
                                    if (referencedModule in dependencyHash) {
                                        // If the circular depenency is against the module that this one is being loaded
                                        // after, ignore it as this module will not be fully attached until after this dependency
                                        // is satisfied.
                                        if (referencedModule == loadAfter) {
                                            continue;
                                        }

                                        throw Error.circularDependency(moduleName, referencedModule);
                                    }

                                    // Attempt to attach the dependent module.
                                    subResult = TFS.attachModule(referencedModule, missingCallback, dependencyHash, visitedHash, loadAfter);
                                    result = (result && subResult);
                                }
                            }

                            // Load any plugins associated with the module.
                            if (attachedPlugins && attachedPlugins.length) {
                                pluginDependencyHash = {};
                                pluginDependencyHash[moduleName] = true;

                                for (i = 0, l = attachedPlugins.length; i < l; i++) {
                                    // Attempt to attach each of the plugin modules providing the current module as the "loadAfter" module.
                                    // This will ensure that erroneous circular dependencies are not reproted and that the module is not fully
                                    // attached during the call (attach will be completed when the current module has been fully attached).
                                    subResult = TFS.attachModule(attachedPlugins[i], missingCallback, pluginDependencyHash, visitedHash, moduleName);
                                    result = (result && subResult);
                                }
                            }
                        }
                        finally {
                            // Now that the processing of this dependency chain is complete, remove the module name from the dependency hash.
                            delete dependencyHash[moduleName];
                        }

                        // If we are not processing a module which should not be loaded until after another module completes
                        // and all of the dependencies are satisfied and the module is not already attached, complete the attach
                        // process.
                        if (!loadAfter && result && !module.attached) {
                            // Everything for the module has been loaded, so complete the initialization of the module.
                            TFS._loadModuleNamespace(moduleName);

                            // Module namespaces will not be loaded for plugins until the module they are dependent upon has been loaded, so perform
                            // the attach of each of the plugin modules now that the main module has been successfuly attached.
                            if (attachedPlugins && attachedPlugins.length) {
                                for (i = 0, l = attachedPlugins.length; i < l; i++) {
                                    TFS.attachModule(attachedPlugins[i], missingCallback, {}, {});
                                }
                            }
                        }
                    }
                    else {
                        // The module has not been downloaded loaded yet, so invoke the missing callback if one was provided.
                        if (missingCallback) {
                            missingCallback(moduleName);
                        }

                        result = false;
                    }

                    visitedHash[moduleName] = result;
                    return result;
                },

                using: function (moduleNames, moduleLoaded) {
                    /// <summary>Enables to have the given modules loaded and ready to use.</summary>
                    /// <param name="moduleNames" type="Array">The list of module names to be used</param>
                    /// <param name="moduleLoaded" type="Function">(Optional) Function to run when
                    /// the module is available</param>

                    var i, l, missing, missingHash, delegate, visitedModules = {};

                    function missingModule(name) {
                        if (!missing) {
                            missing = [];
                            missingHash = {};
                        }

                        if (!(name in modules) && !(name in missingHash)) {
                            missingHash[name] = true;
                            missing.push(name);
                        }
                    }

                    if (moduleNames) {
                        for (i = 0, l = moduleNames.length; i < l; i++) {
                            TFS.attachModule(moduleNames[i], missingModule, null, visitedModules);
                        }
                    }

                    if (missing && missing.length > 0) {
                        // Setup delegate to attempt the using statement again after each of the missing modules has been loaded.
                        delegate = delegateAfterNTimes(TFS, missing.length, function () {
                            TFS.using(moduleNames, moduleLoaded);
                        });

                        // Download the script files for each of the missing modules.
                        for (i = 0, l = missing.length; i < l; i++) {
                            TFS.getScript(missing[i], delegate);
                        }
                    }
                    else {
                        // All of the modules have been loaded, so complete the using by invoking the
                        // module loaded callback if one was provided.
                        if (moduleLoaded) {
                            moduleLoaded.call(TFS);
                        }
                    }
                },
                module: function (namespace, using, body, attach) {
                    /// <summary>Declare JavaScript module and dependent module</summary>
                    /// <param name="namespace" type="String">Unique name of the JavaScript module.  Use dot syntax like TFS.MyArea</param>
                    /// <param name="using" type="Array">Array of strings with the name of dependent modules</param>
                    /// <param name="body" type="function">Function that wraps the entire module to scope variables</param>
                    /// <param name="attach" type="Boolean">Flag indicating if the module should be attached immediately.</param>
                    modules[namespace] = {
                        namespace: namespace,
                        using: using,
                        body: body
                    };

                    if (typeof devTime !== "undefined") {
                        TFS.using(using);
                    }

                    if (attach) {
                        TFS.attachModule(namespace);
                    }
                },
                pluginModule: function (namespace, loadAfter) {
                    var moduleToAttach = modules[loadAfter], attachedPlugins;

                    attachedPlugins = plugins[loadAfter];

                    if (!attachedPlugins) {
                        plugins[loadAfter] = [namespace];
                    }
                    else {
                        if (!attachedPlugins.contains(namespace)) {
                            attachedPlugins.push(namespace);
                        }
                    }

                    if (moduleToAttach && moduleToAttach.attached) {
                        TFS.using([namespace]);
                    }
                },
                queueCallbacks: queueCallbacks,
                queueRequest: queueRequest,
                errorHandler: errorHandler,
                globalProgressIndicator: globalProgressIndicator,
                handleError: handleError,
                getErrorMessage: getErrorMessage
            },
            moduleExceptions);
        }());

        function wrapFunction(fn, superFn, base, baseConstructor) {
            return function () {
                var oldBase = this.base, oldBaseConstructor = this.baseConstructor, oldSuper = this._base;
                this.base = base;
                this.baseConstructor = baseConstructor;

                this._base = function () {
                    return superFn.apply(this, Array.prototype.slice.call(arguments, 0));
                };

                try {
                    return fn.apply(this, Array.prototype.slice.call(arguments, 0));
                }
                finally {
                    this.base = oldBase;
                    this.baseConstructor = oldBaseConstructor;
                    this._base = oldSuper;
                }
            };
        };

        function extendMembers(target, source, base, baseConstructor) {
            var propName, prop;
            for (propName in source) {

                // The type name property should not be cloned since deriving types should have their own unique type name
                if (propName !== "__name" && source.hasOwnProperty(propName)) {
                    prop = source[propName];

                    if (base && typeof prop === "function" && typeof base[propName] === "function") {
                        target[propName] = wrapFunction(prop, base[propName], base, baseConstructor);
                    }
                    else {
                        target[propName] = prop;
                    }
                }
            }

            return target;
        };

        Function.prototype.inherit = function (baseClass, newPrototype) {
            /// <summary>Set up classical style inheritance</summary>
            /// <param name="baseClass" type="Object">Parent object (function)</param>
            /// <param name="newPrototype" type="Object">Child object (function).  Typically an inline function.</param>
            var base, thisPrototype, baseConstructor;

            //Extend static members
            extendMembers(this, $.extend({}, baseClass, this), baseClass, baseClass);
            this.base = baseClass;

            newPrototype = $.extend({}, this.prototype, newPrototype);

            base = baseClass.prototype;
            base.constructor = baseClass;
            baseConstructor = base.baseConstructor;

            function F() { }

            F.prototype = base;
            thisPrototype = new F();

            //Extend regular members
            extendMembers(thisPrototype, newPrototype, base, baseConstructor);

            this.prototype = thisPrototype;
            this.prototype.constructor = this;

            if (baseConstructor) {
                // If there is a base contructor, wrap it so it get's called when child constructor is called
                this.prototype.baseConstructor = wrapFunction(baseClass, baseConstructor, base, baseConstructor);
            }
            else {
                this.prototype.baseConstructor = baseClass;
            }
        };

        Function.prototype.extend = function (staticMembers) {
            /// <summary>Adds static members to functions.</summary>
            /// <param name="staticMembers" type="Object">An object which properties ultimately be static members of the function.</param>
            return extendMembers(this, staticMembers, this.base, this.baseConstructor);
        };

        Function.prototype.inheritBehavior = function (behaviorClass) {
            /// <summary>Adds prototype members of the behaviorClass to this function's prototype.</summary>
            /// <param name="behaviorClass" type="Function">A class which prototype properties ultimately be this functions prototype.</param>
            return extendMembers(this.prototype, behaviorClass.prototype, this.base, this.baseConstructor);
        };

        Function.prototype.curry = function () {
            /// <summary>
            ///     Curries a function with a set of arguments and returns the resulting function.
            ///     When eventually evaluated, the returned function will call the original function
            ///     with the current arguments prepended to the list of arguments.
            ///
            ///     var add3, result;
            ///     function add(x, y) {
            ///         return x + y;
            ///     }
            ///     add3 = add.curry(3);
            ///     results = add3(4); // result === 7
            ///
            ///     See http://en.wikipedia.org/wiki/Curry_function
            /// </summary>
            /// <param name="*" type="*">Any arguments supplied to this function will be "applied" to the calling function</param>
            /// <returns type="Function" />
            var slice = Array.prototype.slice,
            args = slice.apply(arguments),
            that = this;
            return function () {
                return that.apply(this, args.concat(slice.apply(arguments)));
            };
        };

        Function.prototype.getName = (function () {
            var counter = 0;

            return function () {
                var name, f, pos;

                if (!this.__name) {
                    this.__name = "_type_" + (counter++);

                    if (debug) {
                        name = this.name;

                        if (!name) {
                            f = this.toString();
                            name = f.substring(9, f.indexOf("("));
                        }

                        if (name) {
                            this.__name += "_" + name;
                        }
                    }
                }

                return this.__name;
            };
        }());

        function defaultComparer(a, b) {
            //weak equality should be enough for default comparer
            if (a == b) {
                return 0;
            }
            else if (a > b) {
                return 1;
            }
            else {
                return -1;
            }
        }

        Array.prototype.contains = function (value, comparer) {
            var i, l;

            if (typeof value === "undefined") {
                return false;
            }

            comparer = comparer || defaultComparer;

            for (i = 0, l = this.length; i < l; i++) {
                if ((typeof this[i] !== "undefined") && comparer(this[i], value) === 0) {
                    return true;
                }
            }

            return false;
        };

        Array.prototype.intersect = function (array, comparer) {
            var result, i, l, value;

            if (!array) {
                return this;
            }

            if (array.length === 0) {
                return [];
            }

            result = [];

            for (i = 0, l = this.length; i < l; i++) {
                value = this[i];
                if (array.contains(value, comparer)) {
                    result[result.length] = value;
                }
            }

            return result;
        };

        Array.prototype.union = function (array, comparer) {
            var result;

            if (!array || array.length === 0) {
                return this;
            }

            result = this.concat(array);
            result.uniqueSort(comparer);

            return result;
        };

        Array.prototype.uniqueSort = function (comparer) {
            /// <summary>Sorts and removes duplicate elements</summary>
            /// <returns type="Array" />
            var i, l;
            comparer = comparer || defaultComparer;

            this.sort(comparer);

            for (i = 1, l = this.length; i < l; i++) {
                if (comparer(this[i], this[i - 1]) === 0) {
                    this.splice(i--, 1);
                    l--;
                }
            }

            return this;
        };

        Array.prototype.unique = function (comparer) {
            var result = this.slice(0);
            result.uniqueSort(comparer);

            return result;
        };

        Array.prototype.subtract = function (array, comparer) {
            var i, l, value, result;

            if (!array || array.length === 0) {
                return this;
            }

            result = [];

            for (i = 0, l = this.length; i < l; i++) {
                value = this[i];
                if (!array.contains(value, comparer)) {
                    result[result.length] = value;
                }
            }

            return result;
        };

        Array.prototype.reorder = function (oldIndex, newIndex, count) {
            /// <summary>Reorders an array by moving oldIndex + the "count" next elements to the newIndex in the array</summary>
            /// <param name="oldIndex" type="Number">The index of the array element to move</param>
            /// <param name="newIndex" type="Number">The index of the array to insert the element at</param>
            /// <param name="count" type="Number">The number of subsequent, contiguous elements to take with the oldIndex in the reorder</param>

            var move;

            if (newIndex > oldIndex) {
                if ((newIndex - oldIndex) < count) {
                    throw Error.create("Array cannot be reordered if newIndex is within the items being moved");
                }
                newIndex -= count;
            }

            move = this.splice(oldIndex, count);

            Array.prototype.splice.apply(this, [newIndex, 0].concat(move));

            return this;
        };

        Array.flagSorted = function (array, comparer) {
            array.sorted = true;
            array.comparer = comparer;
        };

        Array.copySortFlag = function (toArray, fromArray) {
            toArray.sorted = fromArray.sorted;
            toArray.comparer = fromArray.comparer;
        };

        Array.isSorted = function (array, comparer) {
            return array.sorted && array.comparer === comparer;
        };

        Array.sortIfNotSorted = function (array, comparer) {
            if (!Array.isSorted(array, comparer)) {
                array.sort(comparer);
                Array.flagSorted(array, comparer);
            }
        };

        // Stable reverse means we will keep the order for the equal items after reverse.
        // The motivation is to represent consistent experiences.
        // e.g.
        //  the input array is: [item1, item2, item3, item4, item5]
        //  (item1, item2, and item3 are equal)
        //
        //  after simple reverse: [item5, item4, item3, item2, item1]
        //  after stable reverse: [item5, item4, item1, item2, item3]
        Array.stableReverse = function(array, comparer) {
            var result = [];

            for (var index = array.length - 1; index >= 0; index--) {
                // Try to find the range of the sequential equal items, so we just need to find
                // the index of the first different item
                var firstDiffIndex = index - 1;
                for (; firstDiffIndex >= 0; firstDiffIndex--) {
                    if (0 !== comparer(array[firstDiffIndex], array[index])) {
                        break;
                    }
                }

                for (var equalIndex = firstDiffIndex + 1; equalIndex <= index; equalIndex++) {
                    result.push(array[equalIndex]);
                }

                index = firstDiffIndex + 1;
            }

            for (var index = 0; index < result.length; index++) {
                array[index] = result[index];
            }
        };

        Date.prototype.isMinDate = function () {
            /// <summary>Checks whether this date object corresponds to a min date or not</summary>
            /// <returns type="Boolean" />
            var utcTime = this.getTime() - this.getTimezoneOffset() * 60000;
            return utcTime === -62135596800000; // this constant is utc min date value
        };


        Date.prototype.compare = function (date) {
            /// <summary>Compares this date object with the given date object</summary>
            /// <param name="date" type="Date">Date object to compare</param>
            /// <returns type="Number" />

            if (date instanceof Date) {
                return date - this;
            }
            return -1;
        };

        Date.prototype.equals = function (date) {
            /// <summary>Checks whether the given date object value is same
            /// to this date's value</summary>
            /// <param name="date" type="Date">Date object to check the value</param>
            /// <returns type="Boolean" />

            return this.compare(date) === 0;
        };

        Date.equals = function (date1, date2) {
            /// <summary>Compare two dates to see if they are equal - returning <c>true</c> if they are equal.</summary>
            /// <param name="date1">The first value to compare</param>
            /// <param name="date2">The second value to compare</param>
            /// <returns type="Boolean" />
            if (date1 === null || date1 === undefined) {
                return date1 === date2;
            }
            else {
                return (date1 instanceof Date) && date1.equals(date2);
            }
        };


        var _oldExpandFormat = Date._expandFormat;
        Date._expandFormat = function (dtf, format) {
            if (format === "G") {
                return _oldExpandFormat(dtf, "d") + " " + _oldExpandFormat(dtf, "T");
            }
            else {
                return _oldExpandFormat(dtf, format);
            }
        };

        // There is a bug in the function that get the era year in MS.Ajax that cause it to use Gregorian year all the time and ignore
        // any culture specific year conversion.
        // We are overriding the method here to address this bug
        Date._getEraYear = function Date$_getEraYear(date, dtf, era, sortable) {
            var year = date.getFullYear(),
                convert = dtf.Calendar.convert;
            if (convert) { // if we have a locale conversion function then we need to use it to get the year before applying the era
                year = convert.fromGregorian(date)[0];
            }
            if (!sortable && dtf.eras) {
                year -= dtf.eras[era + 3];
            }
            return year;
        }

        // Adds "G" format (ShortDate + LongTime) to list of dateTimeFormats
        var _oldGetDateTimeFormats = Sys.CultureInfo.prototype._getDateTimeFormats;
        Sys.CultureInfo.prototype._getDateTimeFormats = function () {
            if (!this._myDateFormats) {
                this._myDateFormats = _oldGetDateTimeFormats.apply(this, Array.prototype.slice.call(arguments, 0)).concat([this.dateTimeFormat.ShortDatePattern + ' ' + this.dateTimeFormat.LongTimePattern]);
            }

            return this._myDateFormats;
        }

        // Re-implement Date.parseLocale from MicrosoftAjax.js to account for server's timezone
        var _oldParseLocale = Date.parseLocale;
        Date.parseLocale = function Date$parseLocale(value, formats, ignoreTimeZone) {
            var localDateTime = _oldParseLocale(value, formats);
            if (ignoreTimeZone || !localDateTime) {
                return localDateTime;
            }
            else {
                // Get current time in milliseconds according to client
                var localTime = localDateTime.getTime();

                // Get client's timezone offset in milliseconds
                var localOffset = localDateTime.getTimezoneOffset() * 60000;

                // Adjust client's time to utc, and then adjust to appropriate timezone
                var newUtc = localTime - localOffset - utcOffset;
                var newDateTime = new Date(newUtc);

                return newDateTime;
            }
        }

        // Re-implement Date.prototype.localeFormat from MicrosoftAjax.js to account for server's timezone
        var oldLocaleFormat = Date.prototype.localeFormat;
        Date.prototype.localeFormat = function (format, ignoreTimeZone) {
            if (ignoreTimeZone) {
                return oldLocaleFormat.call(this, format);
            }
            else {
                // Get current time in milliseconds according to client
                var localTime = this.getTime();

                // Get client's timezone offset in milliseconds
                var localOffset = this.getTimezoneOffset() * 60000;

                // Adjust client's time to utc, and then adjust to appropriate timezone
                var newUtc = localTime + localOffset + utcOffset;
                var newDateTime = new Date(newUtc);

                return oldLocaleFormat.call(newDateTime, format);
            }
        }

        function zeroPad(str, count, left) {
            for (var l = str.length; l < count; l++) {
                str = (left ? ('0' + str) : (str + '0'));
            }
            return str;
        }

        Number.prototype.toDecimalLocaleString = function (includeGroupSeparators, cultureInfo) {

            /// <summary>Converts this number to a string in the current culture's locale
            /// without specifying a precision. So, for example, with Spanish culture,
            /// (3) gets translated to "3", and (3.1416) becomes "3,1416". The jQuery's
            /// localeFormat requires a precision (the default is "2" if not specified).
            /// So 3.localeFormat("N") become "3,00".
            /// <param name="includeGroupSeparators" type="bool">If true, use locale-specific
            /// group separators (i.e. 3,000) in the output</param>
            /// <param name="cultureInfo" type="Sys.CultureInfo">Culture info (CurrentCulture if not specified)</param>
            /// <returns type="String" />

            var exponent, nf, split,
                numberString = this.toString(),
                right = "";

            if (cultureInfo) {
                nf = cultureInfo.numberFormat;
            }
            else {
                nf = Sys.CultureInfo.CurrentCulture.numberFormat;
            }

            split = numberString.split(/e/i);
            numberString = split[0];
            exponent = (split.length > 1 ? parseInt(split[1], 10) : 0);
            split = numberString.split('.');
            numberString = split[0];
            right = split.length > 1 ? split[1] : "";

            if (exponent > 0) {
                right = zeroPad(right, exponent, false);
                numberString += right.slice(0, exponent);
                right = right.substr(exponent);
            }
            else if (exponent < 0) {
                exponent = -exponent;
                numberString = zeroPad(numberString, exponent + 1, true);
                right = numberString.slice(-exponent, numberString.length) + right;
                numberString = numberString.slice(0, -exponent);
            }

            if (right.length > 0) {
                right = nf.NumberDecimalSeparator + right;
            }

            if (includeGroupSeparators === true) {
                var groupSizes = nf.NumberGroupSizes,
                    sep = nf.NumberGroupSeparator,
                    curSize = groupSizes[0],
                    curGroupIndex = 1,
                    stringIndex = numberString.length - 1,
                    ret = "";

                while (stringIndex >= 0) {
                    if (curSize === 0 || curSize > stringIndex) {
                        if (ret.length > 0) {
                            return numberString.slice(0, stringIndex + 1) + sep + ret + right;
                        }
                        else {
                            return numberString.slice(0, stringIndex + 1) + right;
                        }
                    }
                    if (ret.length > 0) {
                        ret = numberString.slice(stringIndex - curSize + 1, stringIndex + 1) + sep + ret;
                    }
                    else {
                        ret = numberString.slice(stringIndex - curSize + 1, stringIndex + 1);
                    }
                    stringIndex -= curSize;
                    if (curGroupIndex < groupSizes.length) {
                        curSize = groupSizes[curGroupIndex];
                        curGroupIndex++;
                    }
                }
                return numberString.slice(0, stringIndex + 1) + sep + ret + right;
            }
            else {
                return numberString + right;
            }
        };

        String.prototype.htmlEncode = function () {
            ///	<summary>
            ///		HTML Encodes the string. Use this method to help prevent cross site scripting attacks
            ///     by cleaning text which may contain HTML elements before the string is display in a web page.
            ///	</summary>
            ///	<returns type="String">A copy of the current string which has been HTML encoded</returns>

            // we can write a slightly faster (30%) version that uses a regexp to find and replace
            // all occurences of [&<>"] by their corresponding HTML entities but this approach
            // uses the DOM implementation, so presumably it covers all input/edge cases.
            var div = document.createElement("div");
            div.appendChild(document.createTextNode(this));
            return div.innerHTML;
        };

        String.prototype.nl2br = function () {
            ///	<summary>
            ///		HTML encodes the string and replaces newlines with HTML break tags.
            ///		Use this method to maintain line breaks when displaying strings.
            ///	</summary>
            ///	<returns type="String">A copy of the current string which has been HTML encoded</returns>

            return this.htmlEncode().replace(/(\r\n|\n|\r)/gm, '<br/>');
        };

        String.defaultComparer = function (a, b) {
            var a1 = convertToString(a, false, false),  // Convert the arguments to strings without adjusting casing or locale.
                b1 = convertToString(b, false, false);

            if (a1 === b1) {
                return 0;
            }
            else if (a1 > b1) {
                return 1;
            }
            else {
                return -1;
            }
        };

        String.ignoreCaseComparer = function (a, b) {
            var a1 = convertToString(a, true, false),   // Convert the arguments to upper case strings without adjusting the locale.
                b1 = convertToString(b, true, false);

            if (a1 === b1) {
                return 0;
            }
            else if (a1 > b1) {
                return 1;
            }
            else {
                return -1;
            }
        };

        String.localeComparer = function (a, b) {
            var a1 = convertToString(a, false, true),   // Convert the arguments to locale strings without adjusting the casing.
                b1 = convertToString(b, false, true);

            return a1.localeCompare(b1);
        };

        String.localeIgnoreCaseComparer = function (a, b) {
            var a1 = convertToString(a, true, true),    // Convert the arguments to upercase locale strings.
                b1 = convertToString(b, true, true);

            return a1.localeCompare(b1);
        };

        String.startsWith = function (str, prefix, comparer) {
            comparer = comparer || String.defaultComparer;

            return comparer(prefix, str.substr(0, prefix.length)) === 0;
        };

        function convertToString(value, upperCase, useLocale) {
            /// <summary>Converts the value provided to a string.</summary>
            /// <param name="upperCase" type="boolean">Indicates if the string should be made upper case.</param>
            /// <param name="useLocale" type="boolean">Indicates if the locale should be used in the string conversion.</param>
            var result;

            if (value === null || value === undefined) {
                return "";
            }

            // Convert the value to a string.
            result = useLocale ? value.toLocaleString() : value.toString();

            // Convert the value to upper case if requested.
            if (upperCase) {
                result = useLocale ? result.toLocaleUpperCase() : result.toUpperCase();
            }

            return result;
        }
    }

    Sys._Application.prototype._serializeState = function Patch$Sys$_Application$_serializeState(state) {
        var serverState;
        if ("__s" in state) {
            state = $.extend({}, state);
            serverState = state["__s"];
            delete state["__s"];
        }

        return $.param(state) + (serverState ? "&&" + serverState : "");
    };


    Sys._Application.prototype._deserializeState = function Patch$Sys$_Application$_deserializeState(entry) {
        var result = {};
        entry = entry || '';
        var serverSeparator = entry.indexOf('&&');
        if ((serverSeparator !== -1) && (serverSeparator + 2 < entry.length)) {
            result.__s = entry.substr(serverSeparator + 2);
            entry = entry.substr(0, serverSeparator);
        }

        var tokens = entry.split('&');

        for (var i = 0, l = tokens.length; i < l; i++) {
            var token = tokens[i];
            var equal = token.indexOf('=');
            if ((equal !== -1) && (equal + 1 <= token.length)) {
                var name = token.substr(0, equal);
                var value = token.substr(equal + 1).replace(regexPlus, "%20");
                try {
                    result[name] = decodeURIComponent(value);
                } catch (error) {
                    result[name] = value;
                }
            }
        }

        return result;
    };

}(jQuery, window));

// SIG // Begin signature block
// SIG // MIIajwYJKoZIhvcNAQcCoIIagDCCGnwCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFJjTMYDewYgZ
// SIG // WbsUNREu+utqpIYRoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSCMIIEfgIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGkMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTriJDjAeUgFzrNY4kePTtY
// SIG // l+U3nTBEBgorBgEEAYI3AgEMMTYwNKAagBgAVABGAFMA
// SIG // LgBkAGUAYgB1AGcALgBqAHOhFoAUaHR0cDovL21pY3Jv
// SIG // c29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA58SRearH
// SIG // VkE/BO7dFi68iyT2ZaKbmjYg18wX3JphOCceAJz35wec
// SIG // 8BIsC0i/LWHMV5kayfOO/pfOD5fCwlhNGE0zPFiU4XH7
// SIG // ZGBXUlkaQGjC3/YVH0Xs/7ZwpIjnSEVTy5p9wXsOo+ns
// SIG // d2kfR5I1632ZADCN4P77GQK+8+qGPtrLTy7UKs9Z+H/T
// SIG // rskWTUIzts9ffK6/0gzDksIfi+sIvXUmzEjZWwLWoxLZ
// SIG // jJDtvRpvgeOSl+3mXBXOUQfetSPaZr6jqc3s0vZ1BTMy
// SIG // 6Z5E9WxqKz3oMAirxsTL7i2FqQamsLmvmxrfO43NGZdN
// SIG // v4XofW9XfYzcjReREYHGenjdC6GCAh8wggIbBgkqhkiG
// SIG // 9w0BCQYxggIMMIICCAIBATCBhTB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0ECCmECkkoAAAAAACAwCQYFKw4DAhoFAKBd
// SIG // MBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZI
// SIG // hvcNAQkFMQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZIhvcN
// SIG // AQkEMRYEFO7J5n9eTr5nUy2zKI91+L9EqHQ+MA0GCSqG
// SIG // SIb3DQEBBQUABIIBALGNj+4+OgQHMuh/iMhZGX5FOcpI
// SIG // hTMGT0tZ1Y4XnCs59hYig9RhOkMDdGHtTXz5npgDSW4r
// SIG // WZEOTY0xdizYUqw33PFDZ+3Jocj/xLC1zWyASyNGEB3V
// SIG // t9kv5nfICWzTL6fHOCQ/DNo+pRJx8ysUxjUipDrDQwRO
// SIG // ojYO3XcWpgNGOYBXzCjR676o0NWMi3h6EqewkFb2xI80
// SIG // UFBgVYvcO1PWkcvzrkdp6SZgce3Nzis8GUOujoI376Kr
// SIG // d1kLiaCqQgrOhr6jQwPIAzG6GydhDAPuPI+ofuztA6pl
// SIG // iMhDx3uOue46UidydYjw6IWMMmviJ/LjlZNw8oI1EtQU
// SIG // B8s8yOM=
// SIG // End signature block
